/**
 * File: options.js
 * Purpose: Wire up the options page for the n8n AI Assistant Chrome extension so users can persist model settings.
 * Docs: https://developer.chrome.com/docs/extensions/reference/storage/, https://developer.chrome.com/docs/extensions/mv3/options/
 * Sample Input: { provider: "openai", apiBase: "https://api.openai.com/v1", model: "gpt-4o", allowedSites: ["https://n8n.example.com"] }
 * Expected Output: chrome.storage.sync contains the provided values and the form reflects them after reload.
 */

const DEFAULTS = {
  provider: 'openai',
  apiBase: 'https://api.openai.com/v1',
  model: 'gpt-4o',
  temperature: 0.2,
  createMethod: 'ui',
  allowedSites: ['*'],
};

const STORAGE_KEYS = [...new Set([...Object.keys(DEFAULTS), 'apiKey'])];
const STATUS_TIMEOUT_MS = 4500;

function $(id) {
  return document.getElementById(id);
}

function readNumber(value, fallback) {
  const parsed = Number(value);
  return Number.isFinite(parsed) ? parsed : fallback;
}

function normaliseSites(multilineValue) {
  return multilineValue
    .split(/\n+/)
    .map((entry) => entry.trim())
    .filter(Boolean);
}

function renderSites(list) {
  return (Array.isArray(list) ? list : []).join('\n');
}

function showStatus(kind, message) {
  const el = $('status');
  if (!el) return;
  el.textContent = message;
  el.classList.remove('success', 'error');
  if (kind) el.classList.add(kind);
  if (showStatus.timer) clearTimeout(showStatus.timer);
  if (message) {
    showStatus.timer = setTimeout(() => {
      el.textContent = '';
      el.classList.remove('success', 'error');
    }, STATUS_TIMEOUT_MS);
  }
}

async function loadForm() {
  const cfg = await chrome.storage.sync.get(STORAGE_KEYS);
  const data = { ...DEFAULTS, ...cfg };

  $('provider').value = data.provider;
  $('apiBase').value = data.apiBase;
  $('model').value = data.model;
  $('apiKey').value = cfg.apiKey || '';
  $('temperature').value = String(data.temperature);
  $('createMethod').value = data.createMethod;
  $('allowedSites').value = renderSites(data.allowedSites);
  showStatus('success', 'Settings loaded.');
}

async function saveForm() {
  try {
    const payload = {
      provider: $('provider').value.trim() || DEFAULTS.provider,
      apiBase: $('apiBase').value.trim() || DEFAULTS.apiBase,
      model: $('model').value.trim() || DEFAULTS.model,
      apiKey: $('apiKey').value.trim(),
      temperature: readNumber($('temperature').value, DEFAULTS.temperature),
      createMethod: $('createMethod').value.trim() || DEFAULTS.createMethod,
      allowedSites: normaliseSites($('allowedSites').value),
    };

    if (!payload.apiBase.startsWith('http')) {
      throw new Error('API base must start with http(s)://');
    }

    await chrome.storage.sync.set(payload);
    showStatus('success', 'Settings saved.');
  } catch (error) {
    console.error('Failed to save options', error);
    showStatus('error', String(error?.message || error));
  }
}

async function resetForm() {
  try {
    await chrome.storage.sync.set({ ...DEFAULTS, apiKey: '' });
    await loadForm();
    showStatus('success', 'Defaults restored.');
  } catch (error) {
    console.error('Failed to reset options', error);
    showStatus('error', String(error?.message || error));
  }
}

function init() {
  $('save').addEventListener('click', saveForm);
  $('reset').addEventListener('click', resetForm);
  loadForm();
}

document.addEventListener('DOMContentLoaded', init);
